/*
Impulswahlverfahren zu Mehrfrequenzwahlverfahren Umwandler

uC:  Atmega328p
CLK:  12Mhz  ext. Quarz   

Timer0:  1/4ms Takt   12000000 1/64, OCR0A=47 -> 4000Hz, 0.25ms
Timer1: 8-bit non-inverting Phase
  
DTMF output:
  PB1:  PWM-Generiertes Sinus Signal 
*/

//  Includes / Definitionen
#include <avr/io.h>
#include <avr/interrupt.h>
#include <avr/pgmspace.h>
#include <avr/eeprom.h>
#include "lib-iwv.c"
#include "lib-mfv.c"
#include <util/delay.h>
 
// Globale Variablen Definieren
volatile uint8_t soft_presc0 = 0; // Software Prescaler for Timer 0 (Teilung 1/4 auf 1ms Takt)
volatile uint8_t soft_presc1 = 0; // Software Prescaler for Timer 0 (Teilung 1/100 auf 1/10s Takt)
volatile uint8_t timeout_counter= 0; // Timer-Variable (in s/10) 
volatile uint8_t playback = 0; // Timer-Variable (in ms) for sound playback
volatile uint8_t timer_tast = 0; // Timer-Variable (in ms) for die Tastenentprellung


//  Timer 0 Capture Interrupt
ISR(TIMER0_COMPA_vect){
  // Software Prescaler0
  if (soft_presc0 >= 3){   // 3 ms
    iwv_timer();           // Counter for 1ms Takt
    if (playback < 255){   // 255ms
      playback++;          // Timer1 bis max. 255ms hochzählen
    }
    if (timer_tast < 255){
      timer_tast++;        // Tastenentprelltimer bis max. 255ms hochzählen
    }
    // Software Prescaler1
    if (soft_presc1 >= 100 && timeout_counter < 255){
      timeout_counter++;   // Timeout Counter bis max 255ms hochzählen
      soft_presc1=0;       // Prescaler Counter Zurücksetzen
    } else {
      soft_presc1++;       // Prescaler Counter hochzählen
    }
    soft_presc0=0;         // Prescalercounter0 zurücksetzen
  } else {
    soft_presc0++;         // Prescaler Counter hochzählen
  }
}

//  Timer 1 Overflow Interrupt
ISR(TIMER1_OVF_vect){
  mfv_interrupt();      // MFV Interruptroutine ausführen
}

// Initialization 
void init (void) 
{
  // Port D initialization   
  PIND = 0x00;       //make port as input
  PORTD |= _BV(PD5) | _BV(PD4) | _BV(PD3) | _BV(PD2) | _BV(PD1);  //keep all high
  
  // Port B initialization
  DDRB   |= _BV(PB1);    // PB1 (OC1A) make as output
  PORTB  &= ~_BV(PB1);   // clears PB1
  
  // Port C initialization
  DDRC   |= _BV(PINC0);   // for takt PC0 output takt 1ms testing
  
  // Timer0 configuration
  TIMSK0 |= _BV(OCIE0A);  // Interrupt freigeben
  OCR0A  = 47;            // Timer TOP setzen
  TCCR0B |= _BV(CS01) | _BV(WGM01);  // Gives CLK/8 clock
  
  // Int T1 Overflow enabled
  TIMSK1 |= _BV(TOIE1);   // Enable Compare Match interrupt
  sei();                  // Enable global interrupts
} 


/***************************************************************
  Main
*/
int main(void) 
{
  // Variablen Definieren
  uint8_t dialedNumber=0;  
  uint8_t tast=0;           
  uint8_t dialNumber=0;     
  uint8_t dialedNumber_state=0; // (0: do nothing, 1: PWM playback, 2: Flash-Hook, 3: WWH)
  uint8_t tmp=0;                
  uint8_t eeAddressPointer=0;   // EEPROM address pointer for WWH (Maximal 128)
  uint8_t wwstate=0;            // status of WWH 
  // wwstate: (0: Vor der WWH, 1: neuen Wert setzen, 2: Warten)
  
  // Standard digits 0-9, *, #
  const unsigned char auc_frequency [13][2] = {
      {0, 0},   // none
	  {87, 61},	// 0
	  {79, 46},	// 1
	  {87, 46},	// 2
	  {96, 46},	// 3
	  {79, 50},	// 4
	  {87, 50},	// 5
	  {96, 50},	// 6
	  {79, 56},	// 7
	  {87, 56},	// 8
	  {96, 56},	// 9
	  {79, 61},	// *
	  {96, 61},	// #
  };
  
  init();     
  //  Main Loop
  unsigned char i=1; 
  while(1)
  {
    //Taster Abfragen
    if (timer_tast >= 50 && !(dialedNumber_state) && !(dialNumber)) 
    { // wait at least 50 ms for debouncing buttons
      // WWH * (PD1)
      if (PIND & (1<<PD1)) {  
        tast |=  _BV(PD1);    
      } else if ((tast & (1<<PD1)) && !(PIND & (1<<PD1))) {
        // after 50ms than
        tast &= ~_BV(PD1);        // clear it
        dialedNumber_state=3;     // set to WWH
      }
      // star * (PD3) (dialNumber 11)
      if (PIND & (1<<PD3)) {      
        tast |=  _BV(PD3);        // Set bit high
      } else if((tast & (1<<PD3)) && !(PIND & (1<<PD3))){
        // after 50ms than
        tast &= ~_BV(PD3);        // clear it
        dialNumber= 11;           // set the dialNumber (*)
      }
      // raute #     (PD4) (dialNumber 12)
      if (PIND & _BV(PD4)) {  
        tast |=  _BV(PD4);        // Speichere das Taster Zustand =1 war
      } else if ((tast & (1<<PD4)) && !(PIND & (1<<PD4))){ 
        // after 50ms than
        tast &= ~_BV(PD4);       
        dialNumber=12;        
      }
      // Erdtaste     (PD5) not used!
      if (PIND & (1<<PD5)) {  
        tast |=  _BV(PD5);        
      } else if ((tast & (1<<PD5)) && !(PIND & (1<<PD5))) {  
        tast &= ~_BV(PD5);        
        dialedNumber_state=2;     
      }
      timer_tast=0;               // Timer zurücksetzen
    }
    
    //  check dial pulse 
    if (tmp=iwv_input((~(PIND) & (1<<PD2)) >> PD2)) {  // NSI an PD2
      dialNumber=tmp;            // Wenn != NULL Speichern
    }

    //DTMF output
    if (dialNumber != 0 || dialedNumber_state != 0) {      
      switch (dialedNumber_state) 
      {
        // do nothing -> new dialNumber
        case 0:  
        dialedNumber=dialNumber;        // dialNumber zwischenspeichern and
        dialNumber=0;                   // reset dialNumber 
        timeout_counter=0;              // reset timeout
        dialedNumber_state=1;           // change status to dialed and ready for PWM playback
        if (eeAddressPointer < 128) {   // Bis max. Stelle 128
          eeprom_write_byte(eeAddressPointer,dialedNumber); 
          // Nächstes Byte im EEPROM mit 0 beschreiben um Nummern Ende zu kennzeichnen
          eeprom_write_byte(eeAddressPointer+1,0);  
          eeAddressPointer++;           // Zeiger für nächstes freies Byte erhöhen
        }
        break;     
        // Generate DTMF tone, duration 100 ms
        case 1: // set PWM variables
        if (!(TCCR1B & (1<<CS10))) { 
          x_SWa = auc_frequency[dialedNumber][0]; // high
          x_SWb = auc_frequency[dialedNumber][1]; // low
          TCCR1A = _BV(COM1A1) | _BV(WGM10);      // non inverting 8Bit PWM
          TCCR1B = _BV(CS10);                     // CLK/1 
          playback = 0;               
        } else if (playback >= 10000 ) {          // Stop DTMF transmitting  
          TCCR1A &= ~_BV(COM1A1) & ~_BV(WGM10);    
          TCCR1B &= ~_BV(CS10);         
          PORTB  &= ~_BV(PB1);     
          // In der WWH
          if (wwstate){
            dialedNumber_state=3;     // ser the Status to WWH
			playback = 0;             // Zeit Stoppen
            wwstate=2;                // WWH warten lassen
          } else {                    // Einzelne dialNumber ausgegeben
            dialedNumber_state=0;     // reset Status and do nothing
            dialedNumber=0;           // dialedNumberwert zurücksetzen
          }
        }
        break;
        case 2: // Flash-Hook (FLASH_ZEIT 80ms Port HIGH ziehen)
        if (!(PORTB & (1<<PB1))) {  // check if PB1 is low
          PORTB |= _BV(PB1);        // Port PB1 HIGH ziehen
          playback=0;               // Zeit Stoppen
        } else if (playback >= 80) {
          PORTB &= ~_BV(PB1);       // Port PB1 LOW setzen
          dialedNumber_state=0;     // reset Status and do nothing 
        }
        break;
        case 3: // star * (WWH)
        // clear address pointer 
        if (wwstate == 0) {         // Falls WWH neu gestartet
          eeAddressPointer=0;       // reset address pointer
          wwstate=1;                // Status erhöhen
        } else if (wwstate == 1) {  // Falls WWH aktiv und nicht wartend
          tmp=eeprom_read_byte(eeAddressPointer);
          if (tmp) {                // Wenn Ende noch nicht erreicht (Byte != 0)
            dialedNumber=tmp;       // dialedNumber setzen
            eeAddressPointer++;     // Pointer weiter setzen
            dialedNumber_state=1;   // change status to dialed and ready for PWM playback
          } else {                  // Wenn Datenende erreicht
            eeAddressPointer =0;    // Pointer zurücksetzen
            wwstate=0;              // WWH Ende speichern
            dialedNumber_state=0;   // reset Status and do nothing 
          }
        } else if (wwstate >= 2 && playback >= 50) {    
          // PAUSE (default 50ms) 
          wwstate=1;                // WWH weiterführen
        }
        break;
        // Wenn doch dialedNumber Status wieder auf 0 setzen
        default: dialedNumber_state=0; // Sollte nie eintreten    
      } // switch
    } // Ausführen wenn in der Wahl
    //Timeouts Überprüfen
    if (timeout_counter >= 50 && wwstate == 0) {
      // Wenn TIMEOUT 50ms abgelaufen UND nicht in der WWH-Aufzeichnung von vorne beginnen
      eeAddressPointer=0;          // clear address pointer
    }
  } // while
  return 0;                       // Wird nie erreicht
}
